/*
 *
 *  $Id: pvrusb2-encoder.c 1442 2006-12-28 02:40:26Z isely $
 *
 *  Copyright (C) 2005 Mike Isely <isely@pobox.com>
 *  Copyright (C) 2004 Aurelien Alleaume <slts@free.fr>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include "compat.h"
#include <linux/device.h>   // for linux/firmware.h
#include <linux/firmware.h>
#include "pvrusb2-util.h"
#include "pvrusb2-encoder.h"
#include "pvrusb2-hdw-internal.h"
#include "pvrusb2-debug.h"

#ifndef PVR2_ENABLE_CX2341XMOD
static u32 pvr_tbl_emphasis [] = {
	[PVR2_CVAL_AUDIOEMPHASIS_NONE] = 0x0,
	[PVR2_CVAL_AUDIOEMPHASIS_50_15] = 0x1,
	[PVR2_CVAL_AUDIOEMPHASIS_CCITT] = 0x3,
};

static u32 pvr_tbl_srate[] = {
	[PVR2_CVAL_SRATE_48] =  0x01,
	[PVR2_CVAL_SRATE_44_1] = 0x00,
	[PVR2_CVAL_SRATE_32] =  0x02,
};

static u32 pvr_tbl_audiobitrate[] = {
	[PVR2_CVAL_AUDIOBITRATE_384] = 0xe,
	[PVR2_CVAL_AUDIOBITRATE_320] = 0xd,
	[PVR2_CVAL_AUDIOBITRATE_256] = 0xc,
	[PVR2_CVAL_AUDIOBITRATE_224] = 0xb,
	[PVR2_CVAL_AUDIOBITRATE_192] = 0xa,
	[PVR2_CVAL_AUDIOBITRATE_160] = 0x9,
	[PVR2_CVAL_AUDIOBITRATE_128] = 0x8,
	[PVR2_CVAL_AUDIOBITRATE_112] = 0x7,
	[PVR2_CVAL_AUDIOBITRATE_96]  = 0x6,
	[PVR2_CVAL_AUDIOBITRATE_80]  = 0x5,
	[PVR2_CVAL_AUDIOBITRATE_64]  = 0x4,
	[PVR2_CVAL_AUDIOBITRATE_56]  = 0x3,
	[PVR2_CVAL_AUDIOBITRATE_48]  = 0x2,
	[PVR2_CVAL_AUDIOBITRATE_32]  = 0x1,
	[PVR2_CVAL_AUDIOBITRATE_VBR] = 0x0,
};

static u32 pvr_tbl_audiomodeextension[] = {
	[PVR2_CVAL_AUDIOMODEEXTENSION_BOUND_4] = 0,
	[PVR2_CVAL_AUDIOMODEEXTENSION_BOUND_8] = 1,
	[PVR2_CVAL_AUDIOMODEEXTENSION_BOUND_12] = 2,
	[PVR2_CVAL_AUDIOMODEEXTENSION_BOUND_16] = 3,
};

static u32 pvr_tbl_mpegaudiomode[] = {
	[PVR2_CVAL_MPEGAUDIOMODE_STEREO] = 0,
	[PVR2_CVAL_MPEGAUDIOMODE_JOINT_STEREO] = 1,
	[PVR2_CVAL_MPEGAUDIOMODE_DUAL] = 2,
	[PVR2_CVAL_MPEGAUDIOMODE_MONO] = 3,
};

static u32 pvr_tbl_streamtype[] = {
	[PVR2_CVAL_STREAMTYPE_PS] = 0,
	[PVR2_CVAL_STREAMTYPE_TS] = 1,
	[PVR2_CVAL_STREAMTYPE_SS] = 2,
	[PVR2_CVAL_STREAMTYPE_DVD] = 10,
	[PVR2_CVAL_STREAMTYPE_VCD] = 11,
	[PVR2_CVAL_STREAMTYPE_SVCD] = 12,
	[PVR2_CVAL_STREAMTYPE_DVDS1] = 13,
	[PVR2_CVAL_STREAMTYPE_DVDS2] = 14,
	[PVR2_CVAL_STREAMTYPE_PESAV] = 3,
	[PVR2_CVAL_STREAMTYPE_PESV] = 5,
	[PVR2_CVAL_STREAMTYPE_PESA] = 7,
};

static u32 pvr_tbl_aspect[] = {
	[PVR2_CVAL_ASPECT_1X1] = 1,
	[PVR2_CVAL_ASPECT_4X3] = 2,
	[PVR2_CVAL_ASPECT_16X9] = 3,
	[PVR2_CVAL_ASPECT_221X1] = 4,
};
#endif


/* Firmware mailbox flags - definitions found from ivtv */
#define IVTV_MBOX_FIRMWARE_DONE 0x00000004
#define IVTV_MBOX_DRIVER_DONE 0x00000002
#define IVTV_MBOX_DRIVER_BUSY 0x00000001


static int pvr2_encoder_write_words(struct pvr2_hdw *hdw,
				    const u32 *data, unsigned int dlen)
{
	unsigned int idx;
	int ret;
	unsigned int offs = 0;
	unsigned int chunkCnt;

	/*

	Format: First byte must be 0x01.  Remaining 32 bit words are
	spread out into chunks of 7 bytes each, little-endian ordered,
	offset at zero within each 2 blank bytes following and a
	single byte that is 0x44 plus the offset of the word.  Repeat
	request for additional words, with offset adjusted
	accordingly.

	*/
	while (dlen) {
		chunkCnt = 8;
		if (chunkCnt > dlen) chunkCnt = dlen;
		memset(hdw->cmd_buffer,0,sizeof(hdw->cmd_buffer));
		hdw->cmd_buffer[0] = 0x01;
		for (idx = 0; idx < chunkCnt; idx++) {
			hdw->cmd_buffer[1+(idx*7)+6] = 0x44 + idx + offs;
			PVR2_DECOMPOSE_LE(hdw->cmd_buffer, 1+(idx*7),
					  data[idx]);
		}
		ret = pvr2_send_request(hdw,
					hdw->cmd_buffer,1+(chunkCnt*7),
					NULL,0);
		if (ret) return ret;
		data += chunkCnt;
		dlen -= chunkCnt;
		offs += chunkCnt;
	}

	return 0;
}


static int pvr2_encoder_read_words(struct pvr2_hdw *hdw,int statusFl,
				   u32 *data, unsigned int dlen)
{
	unsigned int idx;
	int ret;
	unsigned int offs = 0;
	unsigned int chunkCnt;

	/*

	Format: First byte must be 0x02 (status check) or 0x28 (read
	back block of 32 bit words).  Next 6 bytes must be zero,
	followed by a single byte of 0x44+offset for portion to be
	read.  Returned data is packed set of 32 bits words that were
	read.

	*/

	while (dlen) {
		chunkCnt = 16;
		if (chunkCnt > dlen) chunkCnt = dlen;
		memset(hdw->cmd_buffer,0,sizeof(hdw->cmd_buffer));
		hdw->cmd_buffer[0] = statusFl ? 0x02 : 0x28;
		hdw->cmd_buffer[7] = 0x44 + offs;
		ret = pvr2_send_request(hdw,
					hdw->cmd_buffer,8,
					hdw->cmd_buffer,chunkCnt * 4);
		if (ret) return ret;

		for (idx = 0; idx < chunkCnt; idx++) {
			data[idx] = PVR2_COMPOSE_LE(hdw->cmd_buffer,idx*4);
		}
		data += chunkCnt;
		dlen -= chunkCnt;
		offs += chunkCnt;
	}

	return 0;
}


/* This prototype is set up to be compatible with the
   cx2341x_mbox_func prototype in cx2341x.h, which should be in
   kernels 2.6.18 or later.  We do this so that we can enable
   cx2341x.ko to write to our encoder (by handing it a pointer to this
   function).  For earlier kernels this doesn't really matter. */
static int pvr2_encoder_cmd(void *ctxt,
			    int cmd,
			    int arg_cnt_send,
			    int arg_cnt_recv,
			    u32 *argp)
{
	unsigned int poll_count;
	int ret = 0;
	unsigned int idx;
	/* These sizes look to be limited by the FX2 firmware implementation */
	u32 wrData[16];
	u32 rdData[16];
	struct pvr2_hdw *hdw = (struct pvr2_hdw *)ctxt;

#if 0
	{
		char buf[150];
		unsigned int ccnt,bcnt;

		bcnt = scnprintf(buf,sizeof(buf),"Encoder Cmd %02x (",cmd);
		for (idx = 0; idx < arg_cnt_send; idx++) {
			if (idx) {
				ccnt = scnprintf(buf+bcnt,
						 sizeof(buf)-bcnt," ");
				bcnt += ccnt;
			}
			ccnt = scnprintf(buf+bcnt,sizeof(buf)-bcnt,
					 "0x%x",argp[idx]);
			bcnt += ccnt;
		}
		ccnt = scnprintf(buf+bcnt,sizeof(buf)-bcnt,")");
		bcnt += ccnt;
		pvr2_trace(PVR2_TRACE_ENCODER,"%.*s",bcnt,buf);
	}
#endif

	/*

	The encoder seems to speak entirely using blocks 32 bit words.
	In ivtv driver terms, this is a mailbox which we populate with
	data and watch what the hardware does with it.  The first word
	is a set of flags used to control the transaction, the second
	word is the command to execute, the third byte is zero (ivtv
	driver suggests that this is some kind of return value), and
	the fourth byte is a specified timeout (windows driver always
	uses 0x00060000 except for one case when it is zero).  All
	successive words are the argument words for the command.

	First, write out the entire set of words, with the first word
	being zero.

	Next, write out just the first word again, but set it to
	IVTV_MBOX_DRIVER_DONE | IVTV_DRIVER_BUSY this time (which
	probably means "go").

	Next, read back 16 words as status.  Check the first word,
	which should have IVTV_MBOX_FIRMWARE_DONE set.  If however
	that bit is not set, then the command isn't done so repeat the
	read.

	Next, read back 32 words and compare with the original
	arugments.  Hopefully they will match.

	Finally, write out just the first word again, but set it to
	0x0 this time (which probably means "idle").

	*/

	if (arg_cnt_send > (sizeof(wrData)/sizeof(wrData[0]))-4) {
		pvr2_trace(
			PVR2_TRACE_ERROR_LEGS,
			"Failed to write cx23416 command"
			" - too many input arguments"
			" (was given %u limit %u)",
			arg_cnt_send,
			(unsigned int)(sizeof(wrData)/sizeof(wrData[0])) - 4);
		return -EINVAL;
	}

	if (arg_cnt_recv > (sizeof(rdData)/sizeof(rdData[0]))-4) {
		pvr2_trace(
			PVR2_TRACE_ERROR_LEGS,
			"Failed to write cx23416 command"
			" - too many return arguments"
			" (was given %u limit %u)",
			arg_cnt_recv,
			(unsigned int)(sizeof(rdData)/sizeof(rdData[0])) - 4);
		return -EINVAL;
	}


	LOCK_TAKE(hdw->ctl_lock); do {

		wrData[0] = 0;
		wrData[1] = cmd;
		wrData[2] = 0;
		wrData[3] = 0x00060000;
		for (idx = 0; idx < arg_cnt_send; idx++) {
			wrData[idx+4] = argp[idx];
		}
		for (; idx < (sizeof(wrData)/sizeof(wrData[0]))-4; idx++) {
			wrData[idx+4] = 0;
		}

		ret = pvr2_encoder_write_words(hdw,wrData,idx);
		if (ret) break;
		wrData[0] = IVTV_MBOX_DRIVER_DONE|IVTV_MBOX_DRIVER_BUSY;
		ret = pvr2_encoder_write_words(hdw,wrData,1);
		if (ret) break;
		poll_count = 0;
		while (1) {
			if (poll_count < 10000000) poll_count++;
			ret = pvr2_encoder_read_words(hdw,!0,rdData,1);
			if (ret) break;
			if (rdData[0] & IVTV_MBOX_FIRMWARE_DONE) {
				break;
			}
			if (poll_count == 100) {
				pvr2_trace(
					PVR2_TRACE_ERROR_LEGS,
					"***WARNING*** device's encoder"
					" appears to be stuck"
					" (status=0%08x)",rdData[0]);
				pvr2_trace(
					PVR2_TRACE_ERROR_LEGS,
					"Encoder command: 0x%02x",cmd);
				for (idx = 4; idx < arg_cnt_send; idx++) {
					pvr2_trace(
						PVR2_TRACE_ERROR_LEGS,
						"Encoder arg%d: 0x%08x",
						idx-3,wrData[idx]);
				}
				pvr2_trace(
					PVR2_TRACE_ERROR_LEGS,
					"Giving up waiting."
					"  It is likely that"
					" this is a bad idea...");
				ret = -EBUSY;
				break;
			}
		}
		if (ret) break;
		wrData[0] = 0x7;
		ret = pvr2_encoder_read_words(
			hdw,0,rdData,
			sizeof(rdData)/sizeof(rdData[0]));
		if (ret) break;
#if 0
		for (idx = 0; idx < args; idx++) {
			if (rdData[idx] != wrData[idx]) {
				pvr2_trace(
					PVR2_TRACE_DEBUG,
					"pvr2_encoder idx %02x mismatch exp:"
					" %08x got: %08x",
					idx,wrData[idx],rdData[idx]);
			}
		}
#endif
		for (idx = 0; idx < arg_cnt_recv; idx++) {
			argp[idx] = rdData[idx+4];
		}

		wrData[0] = 0x0;
		ret = pvr2_encoder_write_words(hdw,wrData,1);
		if (ret) break;

	} while(0); LOCK_GIVE(hdw->ctl_lock);

	return ret;
}


static int pvr2_encoder_vcmd(struct pvr2_hdw *hdw, int cmd,
			     int args, ...)
{
	va_list vl;
	unsigned int idx;
	u32 data[12];

	if (args > sizeof(data)/sizeof(data[0])) {
		pvr2_trace(
			PVR2_TRACE_ERROR_LEGS,
			"Failed to write cx23416 command"
			" - too many arguments"
			" (was given %u limit %u)",
			args,(unsigned int)(sizeof(data)/sizeof(data[0])));
		return -EINVAL;
	}

	va_start(vl, args);
	for (idx = 0; idx < args; idx++) {
		data[idx] = va_arg(vl, u32);
	}
	va_end(vl);

	return pvr2_encoder_cmd(hdw,cmd,args,0,data);
}

#ifdef PVR2_ENABLE_CX2341XMOD
int pvr2_encoder_configure(struct pvr2_hdw *hdw)
{
	int ret;
	pvr2_trace(PVR2_TRACE_ENCODER,"pvr2_encoder_configure"
		   " (cx2341x module)");
	hdw->enc_ctl_state.port = CX2341X_PORT_STREAMING;
	hdw->enc_ctl_state.width = hdw->res_hor_val;
	hdw->enc_ctl_state.height = hdw->res_ver_val;
	hdw->enc_ctl_state.is_50hz = ((hdw->std_mask_cur &
				       (V4L2_STD_NTSC|V4L2_STD_PAL_M)) ?
				      0 : 1);

	ret = 0;

	if (!ret) ret = pvr2_encoder_vcmd(
		hdw,CX2341X_ENC_SET_NUM_VSYNC_LINES, 2,
		0xf0, 0xf0);

	/* setup firmware to notify us about some events (don't know why...) */
	if (!ret) ret = pvr2_encoder_vcmd(
		hdw,CX2341X_ENC_SET_EVENT_NOTIFICATION, 4,
		0, 0, 0x10000000, 0xffffffff);

	if (!ret) ret = pvr2_encoder_vcmd(
		hdw,CX2341X_ENC_SET_VBI_LINE, 5,
		0xffffffff,0,0,0,0);

	if (ret) {
		pvr2_trace(PVR2_TRACE_ERROR_LEGS,
			   "Failed to configure cx23416");
		return ret;
	}

	ret = cx2341x_update(hdw,pvr2_encoder_cmd,
			     (hdw->enc_cur_valid ? &hdw->enc_cur_state : NULL),
			     &hdw->enc_ctl_state);
	if (ret) {
		pvr2_trace(PVR2_TRACE_ERROR_LEGS,
			   "Error from cx2341x module code=%d",ret);
		return ret;
	}

	ret = 0;

	if (!ret) ret = pvr2_encoder_vcmd(
		hdw, CX2341X_ENC_INITIALIZE_INPUT, 0);

	if (ret) {
		pvr2_trace(PVR2_TRACE_ERROR_LEGS,
			   "Failed to initialize cx23416 video input");
		return ret;
	}

	hdw->subsys_enabled_mask |= (1<<PVR2_SUBSYS_B_ENC_CFG);
	memcpy(&hdw->enc_cur_state,&hdw->enc_ctl_state,
	       sizeof(struct cx2341x_mpeg_params));
	hdw->enc_cur_valid = !0;
	return 0;
}
#endif

#ifndef PVR2_ENABLE_CX2341XMOD
int pvr2_encoder_configure(struct pvr2_hdw *hdw)
{
	int ret = 0, audio, i;
	int mpeg1mode,is_30fps;

	mpeg1mode = ((hdw->streamtype_val == PVR2_CVAL_STREAMTYPE_SS) ||
		     (hdw->streamtype_val == PVR2_CVAL_STREAMTYPE_VCD));
	is_30fps = (hdw->std_mask_cur & (V4L2_STD_NTSC|V4L2_STD_PAL_M)) != 0;

	pvr2_trace(PVR2_TRACE_ENCODER,"pvr2_encoder_configure (native)");

	/* set stream output port.  Some notes here: The ivtv-derived
	   encoder documentation says that this command only gets a
	   single argument.  However the Windows driver for the model
	   29xxx series hardware has been sending 0x01 as a second
	   argument, while the Windows driver for the model 24xxx
	   series hardware has been sending 0x02 as a second argument.
	   Confusing matters further are the observations that 0x01
	   for that second argument simply won't work on the 24xxx
	   hardware, while 0x02 will work on the 29xxx - except that
	   when we use 0x02 then xawtv breaks due to a loss of
	   synchronization with the mpeg packet headers.  While xawtv
	   should be fixed to let it resync better (I did try to
	   contact Gerd about this but he has not answered), it has
	   also been determined that sending 0x00 as this mystery
	   second argument seems to work on both hardware models AND
	   xawtv works again.  So we're going to send 0x00. */
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_SET_OUTPUT_PORT, 2,
				 0x01, 0x00);

	/* set the Program Index Information. We want I,P,B frames (max 400) */
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_SET_PGM_INDEX_INFO, 2,
				 0x07, 0x0190);

	/* NOTE : windows driver sends these */
	/* Mike Isely <isely@pobox.com> 7-Mar-2006 The windows driver
	   sends the following commands but if we do the same then
	   many apps are no longer able to read the video stream.
	   Leaving these out seems to do no harm at all, so they're
	   commented out for that reason. */
#ifdef notdef
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_MISC,4, 5,0,0,0);
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_MISC,4, 3,1,0,0);
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_MISC,4, 8,0,0,0);
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_MISC,4, 4,1,0,0);
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_MISC,4, 0,3,0,0);
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_MISC,4,15,0,0,0);
#endif

	/* Strange compared to ivtv data. */
#if 0
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_SET_NUM_VSYNC_LINES, 2,
				 0x0120, 0x0120);
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_SET_NUM_VSYNC_LINES, 2,
				 0x0131, 0x0131);
#endif
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_SET_NUM_VSYNC_LINES, 2,
				 0xf0, 0xf0);

	/* setup firmware to notify us about some events (don't know why...) */
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_SET_EVENT_NOTIFICATION, 4,
				 0, 0, 0x10000000, 0xffffffff);

	/* set fps to 25 or 30 (1 or 0)*/
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_SET_FRAME_RATE, 1,
				 is_30fps ? 0 : 1);

	/* set encoding resolution */
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_SET_FRAME_SIZE, 2,
				 (mpeg1mode ?
				  hdw->res_ver_val / 2 : hdw->res_ver_val),
				 (mpeg1mode ?
				  hdw->res_hor_val / 2 : hdw->res_hor_val));
	/* set encoding aspect ratio */
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_SET_ASPECT_RATIO, 1,
				 pvr_tbl_aspect[hdw->aspect_val]);

	/* VBI */

	if (hdw->config == pvr2_config_vbi) {
		int lines = 2 * (is_30fps ? 12 : 18);
		int size = (4*((lines*1443+3)/4)) / lines;
		ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_SET_VBI_CONFIG, 7,
					 0xbd05, 1, 4,
					 0x25256262, 0x387f7f7f,
					 lines , size);
//                                     0x25256262, 0x13135454, lines , size);
		/* select vbi lines */
#define line_used(l)  (is_30fps ? (l >= 10 && l <= 21) : (l >= 6 && l <= 23))
		for (i = 2 ; i <= 24 ; i++){
			ret |= pvr2_encoder_vcmd(
				hdw,CX2341X_ENC_SET_VBI_LINE, 5,
				i-1,line_used(i), 0, 0, 0);
			ret |= pvr2_encoder_vcmd(
				hdw,CX2341X_ENC_SET_VBI_LINE, 5,
				(i-1) | (1 << 31),
				line_used(i), 0, 0, 0);
		}
	} else {
		ret |= pvr2_encoder_vcmd(
			hdw,CX2341X_ENC_SET_VBI_LINE, 5,
			0xffffffff,0,0,0,0);
	}

	/* set stream type */
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_SET_STREAM_TYPE, 1,
				 pvr_tbl_streamtype[hdw->streamtype_val]);

	/* set video bitrate */
	ret |= pvr2_encoder_vcmd(
		hdw, CX2341X_ENC_SET_BIT_RATE, 5,
		(hdw->vbr_val == PVR2_CVAL_VBR_CONSTANT ? 1 : 0),
		hdw->videobitrate_val,
		hdw->videopeak_val / 400,
		0,0);

	/* setup GOP structure (GOP size = 0f or 0c, 3-1 = 2 B-frames) */
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_SET_GOP_PROPERTIES, 2,
				 is_30fps ?  0x0f : 0x0c, 0x03);

	/* enable / disable 3:2 pulldown */
	ret |= pvr2_encoder_vcmd(hdw,CX2341X_ENC_SET_3_2_PULLDOWN,1,
				 hdw->pulldown_val ? 1 : 0);

	/* set GOP open/close property (open) */
	ret |= pvr2_encoder_vcmd(hdw,CX2341X_ENC_SET_GOP_CLOSURE,1,0);

	/* set audio stream properties 0x40b9? 0100 0000 1011 1001 */
	audio = (pvr_tbl_srate[hdw->srate_val] << 0 |
		 hdw->audiolayer_val << 2 |
		 pvr_tbl_audiobitrate[hdw->audiobitrate_val] << 4 |
		 pvr_tbl_mpegaudiomode[hdw->mpegaudiomode_val] << 8 |
		 pvr_tbl_audiomodeextension[hdw->audiomodeextension_val] << 10 |
		 pvr_tbl_emphasis[hdw->audioemphasis_val] << 12 |
		 (hdw->audiocrc_val ? 1 : 0) << 14);

	ret |= pvr2_encoder_vcmd(hdw,CX2341X_ENC_SET_AUDIO_PROPERTIES,1,
				 audio);

	/* set dynamic noise reduction filter to manual, Horiz/Vert */
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_SET_DNR_FILTER_MODE, 2,
				 0, 0x03);

	/* dynamic noise reduction filter param */
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_SET_DNR_FILTER_PROPS, 2
				 , 0, 0);

	/* dynamic noise reduction median filter */
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_SET_CORING_LEVELS, 4,
				 0, 0xff, 0, 0xff);

	/* spacial prefiler parameter */
	ret |= pvr2_encoder_vcmd(hdw,
				 CX2341X_ENC_SET_SPATIAL_FILTER_TYPE, 2,
				 0x01, 0x01);

	/* initialize video input */
	ret |= pvr2_encoder_vcmd(hdw, CX2341X_ENC_INITIALIZE_INPUT, 0);

	if (!ret) {
		hdw->subsys_enabled_mask |= (1<<PVR2_SUBSYS_B_ENC_CFG);
	}

	return ret;
}
#endif

int pvr2_encoder_start(struct pvr2_hdw *hdw)
{
	int status;

	/* unmask some interrupts */
	pvr2_write_register(hdw, 0x0048, 0xbfffffff);

	/* change some GPIO data */
	pvr2_hdw_gpio_chg_dir(hdw,0xffffffff,0x00000481);
	pvr2_hdw_gpio_chg_out(hdw,0xffffffff,0x00000000);

	pvr2_encoder_vcmd(hdw,CX2341X_ENC_MUTE_VIDEO,1,
			  hdw->input_val == PVR2_CVAL_INPUT_RADIO ? 1 : 0);

	if (hdw->config == pvr2_config_vbi) {
		status = pvr2_encoder_vcmd(hdw,CX2341X_ENC_START_CAPTURE,2,
					   0x01,0x14);
	} else if (hdw->config == pvr2_config_mpeg) {
		status = pvr2_encoder_vcmd(hdw,CX2341X_ENC_START_CAPTURE,2,
					   0,0x13);
	} else {
		status = pvr2_encoder_vcmd(hdw,CX2341X_ENC_START_CAPTURE,2,
					   0,0x13);
	}
	if (!status) {
		hdw->subsys_enabled_mask |= (1<<PVR2_SUBSYS_B_ENC_RUN);
	}
	return status;
}

int pvr2_encoder_stop(struct pvr2_hdw *hdw)
{
	int status;

	/* mask all interrupts */
	pvr2_write_register(hdw, 0x0048, 0xffffffff);

	if (hdw->config == pvr2_config_vbi) {
		status = pvr2_encoder_vcmd(hdw,CX2341X_ENC_STOP_CAPTURE,3,
					   0x01,0x01,0x14);
	} else if (hdw->config == pvr2_config_mpeg) {
		status = pvr2_encoder_vcmd(hdw,CX2341X_ENC_STOP_CAPTURE,3,
					   0x01,0,0x13);
	} else {
		status = pvr2_encoder_vcmd(hdw,CX2341X_ENC_STOP_CAPTURE,3,
					   0x01,0,0x13);
	}

	/* change some GPIO data */
	/* Note: Bit d7 of dir appears to control the LED.  So we shut it
	   off here. */
	pvr2_hdw_gpio_chg_dir(hdw,0xffffffff,0x00000401);
	pvr2_hdw_gpio_chg_out(hdw,0xffffffff,0x00000000);

	if (!status) {
		hdw->subsys_enabled_mask &= ~(1<<PVR2_SUBSYS_B_ENC_RUN);
	}
	return status;
}


/*
  Stuff for Emacs to see, in order to encourage consistent editing style:
  *** Local Variables: ***
  *** mode: c ***
  *** fill-column: 70 ***
  *** tab-width: 8 ***
  *** c-basic-offset: 8 ***
  *** End: ***
  */
